use Test2::V0;
use Bitcoin::Crypto qw(btc_extprv btc_extpub);
use Bitcoin::Crypto::Util qw(to_format);

# extra tests for BIP86 address generation
# generated by https://gugger.guru/cryptography-toolkit/#!/hd-wallet

my @test_data = (
	{
		mnemonic => 'bike cradle antenna occur essay risk grass thrive cheese patch bargain regret',
		data => [
			{
				bip44 => {
					purpose => 86,
					index => 0,
				},
				account_public =>
					'xpub6BoFHqN87JuSTxytiYmV3pzMvrcsPUdweCccDX6p9vjbCtqtRiyyhTfcSanxuJe8151DuqLrLEXwFUL1SzmMiPSgYgHd6MbWtdMA2pKmdBV',
				public =>
					'xpub6FixkemA83ZZP9GroTypixNcYRFcn6v8Ws4pJSvjwFEwve8rdYwCJiWWLyFjEwizXBdrxK2m2N8KYgd3R7PN9MMx7yvN4R1kxMwnJCUGuzb',
				address => 'bc1ptzkkd9f5m8fmwxl2t03x04wmz5rx3l6lrdl9zup6e0lqsya9vzksg5udlk',
			},
			{
				bip44 => {
					purpose => 86,
					account => 7,
					index => 23,
				},
				account_public =>
					'xpub6BoFHqN87JuSnXsq8bieJNs1VmYy1EDzPDT1Bu2VvVK6UrWY7hUbFPpvY1zMeJSroyfU54eDXgVb1sig4vDrQzp3k2xzKLsvJUFYKgdMfm5',
				public =>
					'xpub6G6Xa7bF2KV65f3GvVX7sX9LPJe8X6qD9MdaZgq3vyPmB7fwTkUV4x1Fde34ehmWiFExx8Z2CddU1XQRLiDfhQyyaNRawy1wvRLbvkY9i2F',
				address => 'bc1p2wv86wuk70njyex56rcj0605096fjum9m7vg2vwr7zed34npqmwqjr5zlj',
			},
			{
				bip44 => {
					purpose => 86,
					account => 500,
					change => 1,
					index => 1500,
				},
				account_public =>
					'xpub6BoFHqN87JupSh37sSSUxNyuNo51SC87CcHMFPbNNBMsy4gX1FmsAYViV5LDMmPy8DobH5BQpppJGzErpdJZy8CCEd7x3PbJktR2bfFESut',
				public =>
					'xpub6GgJ5P3Gf9oaz89yuqfLP1yyCQkvLn1anJWNsGQzf8sRD6n6V1LWMJau3KTh9ZzXL7PzzVK4VQophGm12nhPqtW2sEFGqyZAjJXfH8wX2Cm',
				address => 'bc1p3p43mxmkgu2wtvuycjrmdjqdzta9mvryn6eatcxntkvxhex8fwuqx2axh0',
			},
			{
				bip44 => {
					purpose => 86,
					account => 500,
					change => 0,
					index => 1,
				},
				account_public =>
					'xpub6BoFHqN87JupSh37sSSUxNyuNo51SC87CcHMFPbNNBMsy4gX1FmsAYViV5LDMmPy8DobH5BQpppJGzErpdJZy8CCEd7x3PbJktR2bfFESut',
				public =>
					'xpub6GBNuJYfT8CopVyjkhjLukscMGxUhL4ooerX49hcQD4HnBXp5x5DLX8LA161yLunrQiPqDoTLHxbZnV7aUYYHrVq5SPShPnCtEDUhpv3fNn',
				address => 'bc1pp3slgt24h56wc04u9755yk0wul3qxsq2c4c8m4pmtl2lexlgxc7sjrggxy',
			},
		],
	},
);

for my $case_num (0 .. $#test_data) {
	my $tdata = $test_data[$case_num];

	subtest "testing bip86 derivation, case $case_num" => sub {
		my $test_vector = $tdata->{data};
		my $master_key = btc_extprv->from_mnemonic($tdata->{mnemonic});
		for my $tdata (@$test_vector) {
			my $derived_private = $master_key->derive_key_bip44(%{$tdata->{bip44}});
			my $account_public = btc_extpub->from_serialized([base58 => $tdata->{account_public}]);
			my $derived_public = $account_public->derive_key_bip44(%{$tdata->{bip44}});

			my $pubkey = $derived_private->get_public_key();
			is(to_format [base58 => $pubkey->to_serialized], $tdata->{public}, 'derived from private ok');
			is(to_format [base58 => $derived_public->to_serialized], $tdata->{public}, 'derived from public ok');
			is $pubkey->get_basic_key->get_address, $tdata->{address}, 'address ok';
		}
	};
}

done_testing;

